    #define TRACE
#define TRACE_INSTANTIATION
#define TRACE_BEHAVIOR


using System;
using System.Diagnostics;
using System.IO;
using System.Xml;
using System.Collections;
using System.Collections.Specialized;
using System.Collections.Generic;
using Microsoft.Win32;
using System.Threading;
using System.Runtime.InteropServices;
using System.Security;
using System.Security.Principal;
using System.Text;
using System.Reflection;

namespace CSharpRecipes
{
	public class Diagnostics
    {
        #region "8.1 Zarzdzanie wynikami diagnostycznymi we wdroonych aplikacjach"
        public static void ExamineConfigFile()
        {
            TraceSwitch ts = new TraceSwitch("DatabaseSwitch", "Rejestruje tylko transakcje z baz danych.");
			ts.Level = TraceLevel.Off;
        }
        #endregion

        #region "8.2 Szczegowe zarzdzanie wynikami debugowania (ledzenia)"
        public static void EnableTracing()
		{
			// T metod naley skompilowa warunkowo z wykorzystaniem dyrektywy TRACE. W innym przypadku
			//   kod skompiluje si i uruchomi w sytuacji, gdy sta TRACE jest wyczona
            BooleanSwitch DBSwitch = new BooleanSwitch("DatabaseSwitch", "Przecznik do ledzenia operacji z baz danych");
			Console.WriteLine("DBSwitch Enabled = " + DBSwitch.Enabled);
			Console.WriteLine("Description = " + DBSwitch.Description);
			Console.WriteLine("DisplayName = " + DBSwitch.DisplayName);

            BooleanSwitch UISwitch = new BooleanSwitch("UISwitch", "Przecznik do ledzenia operacji z interfejsem uytkownika");
			Console.WriteLine("UISwitch Enabled = " + UISwitch.Enabled);
			Console.WriteLine("Description = " + UISwitch.Description);
			Console.WriteLine("DisplayName = " + UISwitch.DisplayName);

            BooleanSwitch ExceptionSwitch = new BooleanSwitch("ExceptionSwitch", "Przecznik do ledzenia zgaszanych wyjtkw");
			Console.WriteLine("ExceptionSwitch Enabled = " + ExceptionSwitch.Enabled);
			Console.WriteLine("Description = " + ExceptionSwitch.Description);
			Console.WriteLine("DisplayName = " + ExceptionSwitch.DisplayName);
		}
	    #endregion

        #region "8.3 Tworzenie wasnych klas przecznikw"
        public static void TestMySwitchClass()
		{
            AppSpecificSwitch AppSwitch = new AppSpecificSwitch("MyApplication", "Przecznik specyficzny dla mojej aplikacji");
			Console.WriteLine("Description = " + AppSwitch.Description);
			Console.WriteLine("DisplayName = " + AppSwitch.DisplayName);
			Console.WriteLine("\r\nLevel = " + AppSwitch.Level);
			Console.WriteLine("\r\nDisable = " + AppSwitch.Disable);			
			Console.WriteLine("Note = " + AppSwitch.Note);			
			Console.WriteLine("Warning = " + AppSwitch.Warning);			
			Console.WriteLine("MinorError = " + AppSwitch.MinorError);			
			Console.WriteLine("MediumError = " + AppSwitch.MediumError);			
			Console.WriteLine("CriticalError = " + AppSwitch.CriticalError);			
		}



		public enum AppSpecificSwitchLevel 
		{
			Disable = 0,
			Note = 1,
			Warning = 2,
			MinorError = 3,
			MediumError = 4,
			CriticalError = 5
		}

		public class AppSpecificSwitch : Switch 
		{
			protected AppSpecificSwitchLevel level = 0;

			public AppSpecificSwitch(string displayName, string description) 
				: base(displayName, description)
			{
				this.Level = (AppSpecificSwitchLevel)base.SwitchSetting;
			}


            //  Waciwo Level do odczytu i zapisu
			public AppSpecificSwitchLevel Level 
			{
				get
				{
					return level;
				}
				set
				{
					if (value < AppSpecificSwitchLevel.Disable)
					{
						level = AppSpecificSwitchLevel.Disable;
					}
					else if (value > AppSpecificSwitchLevel.CriticalError)
					{
						level = AppSpecificSwitchLevel.CriticalError;
					}
					else
					{
						level = value;
					}
				}
			}

            //  Waciwoci tylko do odczytu dla wartoci typu wyliczeniowego AppSpecificSwitchLevel.
			public bool Disable
			{
				get
				{
					if (level <= AppSpecificSwitchLevel.Disable)
					{
						return (true);
					}
					else
					{
						return (false);
					}
				}
			}

			public bool Note
			{
				get
				{
					if (level <= AppSpecificSwitchLevel.Note)
					{
						return (true);
					}
					else
					{
						return (false);
					}
				}
			}

			public bool Warning
			{
				get
				{
					if (level <= AppSpecificSwitchLevel.Warning)
					{
						return (true);
					}
					else
					{
						return (false);
					}
				}
			}

			public bool MinorError
			{
				get
				{
					if (level <= AppSpecificSwitchLevel.MinorError)
					{
						return (true);
					}
					else
					{
						return (false);
					}
				}
			}

			public bool MediumError
			{
				get
				{
					if (level <= AppSpecificSwitchLevel.MediumError)
					{
						return (true);
					}
					else
					{
						return (false);
					}
				}
			}

			public bool CriticalError
			{
				get
				{
					if (level <= AppSpecificSwitchLevel.CriticalError)
					{
						return (true);
					}
					else
					{
						return (false);
					}
				}
			}
		}

		#endregion

        #region "8.4 Warunkowa kompilacja blokw kodu"
        // Opis znajduje si w recepturze 8.4 w treci ksiki
        #endregion

        #region "8.5 Jak sprawdzi, czy proces przesta odpowiada?"
        public enum ProcessRespondingState
        {
            Responding,
            NotResponding,
            Unknown
        }

        public static ProcessRespondingState IsProcessResponding(Process process)
        {
	        if (process.MainWindowHandle == IntPtr.Zero)
	        {
                Trace.WriteLine("{0} nie posiada uchwytu MainWindowHandle",
                    process.ProcessName);
                return ProcessRespondingState.Unknown;
	        }
	        else
	        {
                //  Ten proces posiada uchwyt  MainWindowHandle.
		        if (!process.Responding)
		        {
                    Trace.WriteLine("{0} nie odpowiada.", process.ProcessName);
			        return ProcessRespondingState.NotResponding;
		        }
		        else
		        {
                    Trace.WriteLine("{0} odpowiada.", process.ProcessName);
			        return ProcessRespondingState.Responding;
		        }
	        }
        }
        #endregion

        #region "8.6 Wykorzystanie dziennikw zdarze w aplikacji"
        public static void TestEventLogClass()
		{
			// Powoduje wyjtek
			//AppEvents AppEventLog1 = new AppEvents("AppLog", "AppLocal");
			//AppEvents GlobalEventLog1 = new AppEvents("Application", "AppLocal");
			//GlobalEventLog1.WriteToLog("",EventLogEntryType.Information, CategoryType.AppStartUp, EventIDType.ExceptionThrown);
            
            AppEvents AE = null;
            try
            {
                AE = new AppEvents("", "MYSOURCE");
            }
            catch (Exception e)
            {
                Debug.WriteLine(e.ToString());
            }
            AE.WriteToLog("MESSAGE", EventLogEntryType.Information, CategoryType.AppStartUp, EventIDType.ExceptionThrown, new byte[10] { 1, 2, 3, 4, 5, 6, 7, 8, 9, 0 });
            AE.WriteToLog("MESSAGE", EventLogEntryType.Information, CategoryType.ReadFromDB, EventIDType.Read);

            System.Threading.Thread.Sleep(250);

            foreach (EventLogEntry Entry in AE.GetEntries())
            {
                Console.WriteLine("\r\nMessage:        " + Entry.Message);
                Console.WriteLine("Category:       " + Entry.Category);
                Console.WriteLine("CategoryNumber: " + Entry.CategoryNumber);
                Console.WriteLine("EntryType:      " + Entry.EntryType.ToString());
                Console.WriteLine("InstanceId:     " + Entry.InstanceId);
                Console.WriteLine("Index:          " + Entry.Index);
                Console.WriteLine("MachineName:    " + Entry.MachineName);
                Console.WriteLine("Source:         " + Entry.Source);
                Console.WriteLine("TimeGenerated:  " + Entry.TimeGenerated);
                Console.WriteLine("TimeWritten:    " + Entry.TimeWritten);
                Console.WriteLine("UserName:       " + Entry.UserName);

                foreach (byte data in Entry.Data)
                    Console.WriteLine("\tData: " + data);
            }

            AE.ClearLog();
            AE.CloseLog();
            AE.DeleteLog();
			
			
			// CreateMultipleLogs
            AppEvents AppEventLog = new AppEvents("AppLog", "AppLocal");
            AppEvents GlobalEventLog = new AppEvents("System", "AppGlobal");

            ListDictionary LogList = new ListDictionary();
            LogList.Add(AppEventLog.Name, AppEventLog);
            LogList.Add(GlobalEventLog.Name, GlobalEventLog);

            ((AppEvents)LogList[AppEventLog.Name]).WriteToLog("Uruchomienie aplikacji",
                EventLogEntryType.Information, CategoryType.AppStartUp, 
                EventIDType.ExceptionThrown);

            ((AppEvents)LogList[GlobalEventLog.Name]).WriteToLog("Pocztkowy test bezpieczestwa",
                EventLogEntryType.Information, CategoryType.AppStartUp, 
                EventIDType.BufferOverflowCondition);

            foreach (DictionaryEntry Log in LogList)
            {
	            ((AppEvents)Log.Value).WriteToLog("Uruchomienie aplikacji", 
                    EventLogEntryType.FailureAudit, 
		            CategoryType.AppStartUp, EventIDType.SecurityFailure);
            }

            foreach (DictionaryEntry Log in LogList)
            {
	            ((AppEvents)Log.Value).DeleteLog();
            }
            LogList.Clear();

		}



        public enum EventIDType
        {
	        NA = 0,
	        Read = 1,
	        Write = 2,
	        ExceptionThrown = 3,
	        BufferOverflowCondition = 4,
	        SecurityFailure = 5,
	        SecurityPotentiallyCompromised = 6 
        }

        public enum CategoryType : short
        {
	        None = 0,
	        WriteToDB = 1,
	        ReadFromDB = 2,
	        WriteToFile = 3,
	        ReadFromFile = 4,
	        AppStartUp = 5,
	        AppShutDown = 6,
	        UserInput = 7	
        }

        public class AppEvents
        {
            //  Konstruktory
	        public AppEvents(string logName) : 
		        this(logName, Process.GetCurrentProcess().ProcessName, ".") {}

	        public AppEvents(string logName, string source) : this(logName, source, ".") {}

	        public AppEvents(string logName, string source, string machineName)
	        {
		        this.logName = logName;
		        this.source = source;
		        this.machineName = machineName;

                if (!EventLog.SourceExists(source, machineName)) 
                {
                    EventSourceCreationData sourceData = 
                        new EventSourceCreationData(source, logName);
                    sourceData.MachineName = machineName;

	                EventLog.CreateEventSource(sourceData);
                }

		        log = new EventLog(logName, machineName, source);
		        log.EnableRaisingEvents = true;
	        }


	        // Pola
	        private EventLog log = null;
	        private string source = "";
	        private string logName = "";
	        private string machineName = ".";


            // Waciwoci
	        public string Name
	        {
		        get{return (logName);}
	        }

	        public string SourceName
	        {
		        get{return (source);}
	        }

	        public string Machine
	        {
		        get{return (machineName);}
	        }


            // Metody
	        public void WriteToLog(string message, EventLogEntryType type, 
		        CategoryType category, EventIDType eventID)
	        {
		        if (log == null)
		        {	
			        throw (new ArgumentNullException("log",
                        "Tego dziennika zdarze nie otwarto bd go zamknito."));
		        }

		        log.WriteEntry(message, type, (int)eventID, (short)category);
	        }

	        public void WriteToLog(string message, EventLogEntryType type, 
		        CategoryType category, EventIDType eventID, byte[] rawData)
	        {
		        if (log == null)
		        {
			        throw (new ArgumentNullException("log",
                        "Tego dziennika zdarze nie otwarto bd go zamknito."));
		        }

		        log.WriteEntry(message, type, (int)eventID, (short)category, rawData);
	        }

	        public EventLogEntryCollection GetEntries()
	        {
		        if (log == null)
		        {
			        throw (new ArgumentNullException("log",
                        "Tego dziennika zdarze nie otwarto bd go zamknito."));
		        }

		        return (log.Entries);
	        }

	        public void ClearLog()
	        {
		        if (log == null)
		        {
			        throw (new ArgumentNullException("log",
                        "Tego dziennika zdarze nie otwarto bd go zamknito."));
		        }

		        log.Clear();
	        }

	        public void CloseLog()
	        {
		        if (log == null)
		        {
			        throw (new ArgumentNullException("log",
                        "Tego dziennika zdarze nie otwarto bd go zamknito."));
		        }

		        log.Close();
		        log = null;
	        }

	        public void DeleteLog()
	        {
                if (EventLog.SourceExists(source, machineName)) 
                {
	                EventLog.DeleteEventSource(source, machineName);
                }

                if (logName != "Application" && 
	                logName != "Security" &&
	                logName != "System")
                {
	                if (EventLog.Exists(logName, machineName)) 
	                {
		                EventLog.Delete(logName, machineName);
	                }
                }

                if (log != null)
                {
	                log.Close();
	                log = null;
                }
	        }
        }
		#endregion

        #region "8.7 Modyfikacja maksymalnego rozmiaru niestandardowego dziennika zdarze"
        public static void SetCustomLogMaxSize(string logName, int maxSize)
		{
			RegistryKey Key = Registry.LocalMachine.OpenSubKey
				(@"SYSTEM\CurrentControlSet\Services\Eventlog\" + logName, true);
			if (Key == null)
			{
                Console.WriteLine("Klucz rejestru dla tego dziennika zdarze nie istnieje.");
			}
			else
			{
				Key.SetValue("MaxSize", maxSize);
				Registry.LocalMachine.Close();
			}
		}
        #endregion

        #region "8.8 Wyszukiwanie zapisw w dzienniku zdarze"
        public static void FindAnEntryInEventLog()
        {
	        EventLog Log = new EventLog("System");

	        EventLogEntry[] Entries = EventLogSearch.FindEntryType(Log.Entries, 
		        EventLogEntryType.Error);

	        foreach (EventLogEntry Entry in Entries)
	        {
		        Console.WriteLine("Message:        " + Entry.Message);
                Console.WriteLine("InstanceId:        " + Entry.InstanceId);
		        Console.WriteLine("Category:       " + Entry.Category);
		        Console.WriteLine("EntryType:      " + Entry.EntryType.ToString());
		        Console.WriteLine("Source:         " + Entry.Source);
	        }
        }

        public static void FindAnEntryInEventLog2()
        {
	        EventLog Log = new EventLog("System");

	        EventLogEntry[] Entries = EventLogSearch.FindTimeGeneratedAtOrAfter(Log.Entries, 
		        DateTime.Parse("8/3/2003"));
	        Entries = EventLogSearch.FindEntryType(Entries, EventLogEntryType.Error);
	        Entries = EventLogSearch.FindInstanceId(Entries, 7000);

	        foreach (EventLogEntry Entry in Entries)
	        {
		        Console.WriteLine("Message:        " + Entry.Message);
                Console.WriteLine("InstanceId:        " + Entry.InstanceId);
		        Console.WriteLine("Category:       " + Entry.Category);
		        Console.WriteLine("EntryType:      " + Entry.EntryType.ToString());
		        Console.WriteLine("Source:         " + Entry.Source);
	        }
        }
		
		
        public sealed class EventLogSearch
        {
	        // Dostpne (Statyczne) metody wyszukiwania:			
	        //    FindTimeGeneratedAtOrBefore
	        //    FindTimeGeneratedAtOrAfter
	        //    FindUserName
	        //    FindMachineName
	        //    FindCategory
	        //    FindSource
	        //    FindEntryType
	        //    FindMessage
	        //    FindEventID

            private EventLogSearch() { }     // Zablokowanie moliwoci tworzenia egzemplarzy klasy.

	        public static EventLogEntry[] FindTimeGeneratedAtOrBefore(
		        IEnumerable logEntries, DateTime timeGeneratedQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.TimeGenerated <= timeGeneratedQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindTimeGeneratedAtOrAfter(
		        IEnumerable logEntries, DateTime timeGeneratedQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.TimeGenerated >= timeGeneratedQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindUserName(IEnumerable logEntries, 
		        string userNameQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.UserName == userNameQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindMachineName(IEnumerable logEntries, 
		        string machineNameQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.MachineName == machineNameQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindCategory(IEnumerable logEntries, 
		        string categoryQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.Category == categoryQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindCategory(IEnumerable logEntries, 
		        short categoryNumQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.CategoryNumber == categoryNumQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindSource(IEnumerable logEntries, 
		        string sourceQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.Source == sourceQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindEntryType(IEnumerable logEntries, 
		        EventLogEntryType entryTypeQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.EntryType == entryTypeQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindMessage(IEnumerable logEntries, 
		        string messageQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
			        if (logEntry.Message == messageQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }

	        public static EventLogEntry[] FindInstanceId(IEnumerable logEntries, 
		        int instanceIDQuery)
	        {
		        ArrayList entries = new ArrayList();

		        foreach (EventLogEntry logEntry in logEntries)
		        {
                    if (logEntry.InstanceId == instanceIDQuery)
			        {
				        entries.Add(logEntry);
			        }
		        }

		        EventLogEntry[] entriesArray = new EventLogEntry[entries.Count];
		        entries.CopyTo(entriesArray);
		        return (entriesArray);
	        }
        }
        #endregion

        #region "8.9 Obserwacja specyficznego zapisu w dzienniku zdarze"
        public static void WatchForAppEvent(EventLog log)
		{
			log.EnableRaisingEvents = true;
			log.EntryWritten += new EntryWrittenEventHandler(OnEntryWritten);
			
			TestEventLogClass();
		}

		public static void OnEntryWritten(object source, EntryWrittenEventArgs entryArg) 
		{
			if (entryArg.Entry.EntryType == EventLogEntryType.Error)
			{
                // Wykonywanie dalszych dziaa wedug potrzeb...
				Console.WriteLine(entryArg.Entry.Category);
				Console.WriteLine(entryArg.Entry.EntryType.ToString());
				Console.WriteLine(entryArg.Entry.Message);
				Console.WriteLine("Zapisano");
			}
		}
        #endregion

        #region "8.10 Wyszukiwanie wszystkich rde nalecych do okrelonego dziennika zdarze"
        public static List<string> FindSourceNamesFromLog(string logName)
        {
	        List<string> sourceNamesList = new List<string>();

            //  Pobranie klucza rejestru dla okrelonego dziennika.
            RegistryKey keyLog = Registry.LocalMachine.OpenSubKey
                (@"SYSTEM\CurrentControlSet\Services\Eventlog\" + logName);
            if (keyLog != null && keyLog.SubKeyCount>0)
            {
                // Pobranie rde dziennika.
		        string[] sourceNames = keyLog.GetSubKeyNames();

                // Ustawienie objtoci listy.
		        sourceNamesList.Capacity = keyLog.SubKeyCount;

                // Dodanie rde do listy.
                sourceNamesList.AddRange(sourceNames);
	        }

            // Zwrcenie listy.
            return sourceNamesList;
        }

        public static Hashtable FindSourceNamesFromAllLogs()
        {
            // Utworzenie tablicy  hashtable w celu zapisania dziennikw i ich rde.
	        Hashtable logsAndSources = new Hashtable();

            // Pobranie listy wszystkich dziennikw.
	        string[] eventLogNames = Registry.LocalMachine.OpenSubKey 
		        (@"SYSTEM\CurrentControlSet\Services\Eventlog").GetSubKeyNames();
        		
	        foreach (string log in eventLogNames)
	        {
                // Pobranie wszystkich nazw rde dla dziennika.
                List<string> sourceNamesList = FindSourceNamesFromLog(log);

                // Dodanie do tablicy hashtable listy nazw rde z nazw dziennika.
                // jako indeksem.
		        logsAndSources.Add(log, sourceNamesList);
	        }

            // kod obsugi dostpu do dziennika
            foreach (DictionaryEntry DE in logsAndSources)
            {
                Console.WriteLine("Dziennik: " + DE.Key);              // Wywietlenie dziennika
                foreach (string source in ((List<string>)DE.Value))
                {
                    // Wywietlenie wszystkich rde dla dziennika.
                    Console.WriteLine("\trdo: " + source);
                }
            }


	        return logsAndSources;
        }
        #endregion

        #region "8.11 Implementacja prostego licznika wydajnoci"
        public static void TestCreateSimpleCounter()
		{
			PerformanceCounter AppCounter = CreateSimpleCounter("AppCounter","", PerformanceCounterType.CounterTimer, "AppCategory", "");
			AppCounter.RawValue = 10;
			for (int i = 0; i <= 10; i++)
			{
				CounterSample CounterSampleValue = AppCounter.NextSample();
				Console.WriteLine("\r\n--> Przykadowa warto RawValue  = " + CounterSampleValue.RawValue);
				
				long Value = AppCounter.IncrementBy(i * 2);
				System.Threading.Thread.Sleep(10 * i);

				CounterSample CounterNextSampleValue = AppCounter.NextSample();
				Console.WriteLine("--> Nastpna warto RawValue = " + CounterNextSampleValue.RawValue);
				
				Console.WriteLine("Przyrost czasu = " + (CounterNextSampleValue.TimeStamp - CounterSampleValue.TimeStamp));
				Console.WriteLine("Przyrost czasu [100ns] = " + (CounterNextSampleValue.TimeStamp100nSec - CounterSampleValue.TimeStamp100nSec));
				Console.WriteLine("Przyrost CounterTimeStamp = " + (CounterNextSampleValue.CounterTimeStamp - CounterSampleValue.CounterTimeStamp));
				
				float Sample1 = CounterSample.Calculate(CounterSampleValue, CounterNextSampleValue);
				Console.WriteLine("--> Obliczona prbka = " + Sample1);
			}
		}

		public static PerformanceCounter CreateSimpleCounter(string counterName, string counterHelp, 
			PerformanceCounterType counterType, string categoryName, string categoryHelp)
		{
			CounterCreationDataCollection counterCollection = 
				new CounterCreationDataCollection();

            // Utworzenie niestandardowego obiektu licznika i dodanie go do kolekcji licznikw.
			CounterCreationData counter = new CounterCreationData(counterName, counterHelp,
				counterType);
			counterCollection.Add(counter);

            // Utworzenie kategorii.
			if (PerformanceCounterCategory.Exists(categoryName))
			{
				PerformanceCounterCategory.Delete(categoryName);
			}

            PerformanceCounterCategory appCategory = 
            //Ujto w komentarz jako przestarzae
            //PerformanceCounterCategory.Create(categoryName, categoryHelp, counterCollection);
            PerformanceCounterCategory.Create(categoryName, categoryHelp,
                PerformanceCounterCategoryType.SingleInstance, counterCollection);

            // Utworzenie licznika i zainicjowanie go.
			PerformanceCounter appCounter = 
				new PerformanceCounter(categoryName, counterName, false);

			appCounter.RawValue = 0;

			return (appCounter);
		}
        #endregion

        #region "8.12 Implementacja licznikw wydajnoci, ktre wymagaj licznikw bazowych"
        public static void TestCreateComplexCounter()
		{
			PerformanceCounter AppCounter;
			PerformanceCounter MyCounterBase;
			CreateComplexCounter ("AppCounter","", PerformanceCounterType.SampleFraction, "MyCounterBase","", PerformanceCounterType.SampleBase, 
								  "AppCategory", "", out AppCounter, out MyCounterBase);
							
			AppCounter.RawValue = 10;
			MyCounterBase.RawValue = 1;
			
			for (int i = 0; i <= 10; i++)
			{
				CounterSample CounterSampleValue = AppCounter.NextSample();
				//CounterSample CounterSampleValueBase = MyCounterBase.NextSample();
				Console.WriteLine("\r\n--> Przykadowa warto RawValue  = " + CounterSampleValue.RawValue);
				//Console.WriteLine("--> Przykadowa warto BASE-RawValue  = " + CounterSampleValueBase.RawValue);
				
				long Value = AppCounter.IncrementBy(i * 2);
				long ValueBase = MyCounterBase.IncrementBy(1);

				CounterSample CounterNextSampleValue = AppCounter.NextSample();
				//CounterSample CounterNextSampleValueBase = MyCounterBase.NextSample();
				Console.WriteLine("\r\n--> Przykadowa warto RawValue  = " + CounterNextSampleValue.RawValue);
				//Console.WriteLine("--> Przykadowa warto BASE-RawValue  = " + CounterNextSampleValueBase.RawValue);
				
				Console.WriteLine("Przyrost czasu = " + (CounterNextSampleValue.TimeStamp - CounterSampleValue.TimeStamp));
				Console.WriteLine("Przyrost czasu [100ns] = " + (CounterNextSampleValue.TimeStamp100nSec - CounterSampleValue.TimeStamp100nSec));
				Console.WriteLine("Przyrost CounterTimeStamp = " + (CounterNextSampleValue.CounterTimeStamp - CounterSampleValue.CounterTimeStamp));

				float Sample1 = CounterSample.Calculate(CounterSampleValue, CounterNextSampleValue);
				Console.WriteLine("--> Obliczona prbka = " + Sample1);
			}
		}

		public static void CreateComplexCounter(string counterName, string counterHelp, 
			PerformanceCounterType counterType, string baseCounterName, string baseCounterHelp, 
			PerformanceCounterType baseCounterType, string categoryName, string categoryHelp,
			out PerformanceCounter appCounter, out PerformanceCounter appBaseCounter)
		{
			CounterCreationDataCollection counterCollection = new CounterCreationDataCollection();

            // Utworzenie obiektu licznika uytkownika oraz obiektu jego licznika bazowego
            // i dodanie ich do kolekcji licznikw (naley je doda 
            // w odpowiedniej kolejnoci).

			CounterCreationData counter = new CounterCreationData(counterName, counterHelp,
				counterType);
			counterCollection.Add(counter);
			CounterCreationData BaseCounter = new CounterCreationData(baseCounterName, 
				baseCounterHelp, baseCounterType);
			counterCollection.Add(BaseCounter);

            // Utworzenie kategorii.
			if (PerformanceCounterCategory.Exists(categoryName))
			{
				PerformanceCounterCategory.Delete(categoryName);
			}

            PerformanceCounterCategory appCategory = 
	            PerformanceCounterCategory.Create(categoryName, categoryHelp,
                    PerformanceCounterCategoryType.SingleInstance,
                    counterCollection);

            // Utworzenie licznika i zainicjowanie go.
			PerformanceCounter newAppCounter = 
				new PerformanceCounter(categoryName, counterName, false);
			PerformanceCounter newAppBaseCounter = 
				new PerformanceCounter(categoryName, baseCounterName, false);

			newAppCounter.RawValue = 0;
			newAppBaseCounter.RawValue = 0;

			appCounter = newAppCounter;
			appBaseCounter = newAppBaseCounter;
		}
        #endregion

        #region "8.13 Wczanie i wyczanie zoonego kodu ledzenia"
        public static void TestTraceFactoryClass()
		{
			Trace.Listeners.Clear();
			
			TraceFactory Factory = new TraceFactory();
			Fooz Obj = Factory.CreateObj();
			Console.WriteLine(Obj.ToString());
			Console.WriteLine(Obj.GetHashCode());
			Obj.SomeBehavior();
		}


		public class TraceFactory
		{
			public TraceFactory() {}

			public Fooz CreateObj()
			{
				Fooz Obj = null;

#if (TRACE)
#if (TRACE_INSTANTIATION)
				Obj = new BarTraceInst();
#elif (TRACE_BEHAVIOR)
                Obj = new BarTraceBehavior();
#else
                Obj = new Bar();
#endif
#else
            Obj = new Bar();
#endif

				return (Obj);
			}
		}

		public abstract class Fooz
		{
			public virtual void SomeBehavior()
			{
				//...
			}
		}

		public class Barz : Fooz
		{
			public Barz() {}

			public override void SomeBehavior()
			{
				base.SomeBehavior();
			}
		}

		public class BarTraceInst : Fooz
		{
			public BarTraceInst() 
			{
				Trace.WriteLine("Utworzono egzemplarz obiektu BarTraceInst");
			}

			public override void SomeBehavior()
			{
				base.SomeBehavior();
			}
		}

		public class BarTraceBehavior : Fooz
		{
			public BarTraceBehavior() {}

			public override void SomeBehavior()
			{
				Trace.WriteLine("Wywoano metod SomeBehavior");
				base.SomeBehavior();
			}
		}
		#endregion

        #region "8.14 Przechwytywanie standardowego wyniku procesu"
        public static void TestCaptureOutput()
        {
            // wicej informacji na temat przekierowania mona znale w recepturze 12.21...
            Process application = new Process();
            // uruchomienie powoki polece
            application.StartInfo.FileName = @"cmd.exe";

            // Pobranie listingu biecego katalogu.
            application.StartInfo.Arguments = @"/Cdir " + Environment.CurrentDirectory;
            Console.WriteLine("Uruchomienie powoki cmd.exe z argumentami: {0}", application.StartInfo.Arguments);

            // Przekierowanie standardowego wyjcia tak, aby mona byo je odczyta.
            application.StartInfo.RedirectStandardOutput = true;
            application.StartInfo.UseShellExecute = false;

            // Utworzenie pliku dziennika w celu zapisania wynikw w katalogu biecego pliku EXE.
            using (StreamWriter logger = new StreamWriter("cmdoutput.log"))
            {
                // Uruchomienie powoki.
                application.Start();

                // Pobranie strumienia stdout.
                StreamReader output = application.StandardOutput;

                // Zrzucanie strumienia wynikowego w czasie dziaania aplikacji.
                do
                {
                    using (output)
                    {
                        char[] info = null;
                        while (output.Peek() >= 0)
                        {
                            info = new char[4096];
                            output.Read(info, 0, info.Length);
                            // Zapis do dziennika.
                            logger.Write(info, 0, info.Length);
                        }
                    }
                }
                while (!application.HasExited);
            }

            // Zamknicie obiektu procesu.
            application.Close();

        }
		#endregion

        #region "8.15 Tworzenie niestandardowego wyjcia debugowania dla klas uytkownika"
        [DebuggerDisplay("Citizen Full Name = {_honorific}{_first}{_middle}{_last}")]
        public class Citizen
        {
            private string _honorific;
            private string _first;
            private string _middle;
            private string _last;

            public Citizen(string honorific, string first, string middle, string last)
            {
                _honorific = honorific;
                _first = first;
                _middle = middle;
                _last = last;
            }
        }
        public static void TestCustomDebuggerDisplay()
        {
            Citizen mrsJones = new Citizen("Pani", "Alicja", "G.", "Jankowska");
            Citizen mrJones = new Citizen("Pan", "Robert", "Fryderyk", "Jankowski");
        }
		#endregion

        #region "8.16 Odczytywanie ustawie biecej domeny AppDomain"
        public static void ReadAppDomainSettings()
        {
            AppDomainSetup info = AppDomain.CurrentDomain.SetupInformation;
            Console.WriteLine("Waciwoci biecej domeny AppDomain:");
            if (info.ActivationArguments != null)
            {
                if (info.ActivationArguments.ApplicationIdentity != null)
                {
                    Console.WriteLine("\tElement CodeBase domeny AppDomain: {0}",
                        info.ActivationArguments.ApplicationIdentity.CodeBase);
                    Console.WriteLine("\tPena nazwa domeny AppDomain: {0}",
                        info.ActivationArguments.ApplicationIdentity.FullName);
                }
                foreach (string data in info.ActivationArguments.ActivationData)
                {
                    Console.WriteLine("\tDane dotyczce aktywacji: {0}", data);
                }
                if (info.ActivationArguments.ActivationContext != null)
                {
                    Console.WriteLine("\tTosamo domeny AppDomain: {0}",
                        info.ActivationArguments.ActivationContext.Identity);
                }
            }
            Console.WriteLine("\tBaza biecej aplikacji: {0}",
                info.ApplicationBase);
            Console.WriteLine("\tNazwa biecej aplikacji: {0}",
                info.ApplicationName);

            if (info.ApplicationTrust != null)
            {
                Console.WriteLine("\tInformacje dotyczce zabezpiecze aplikacji:");
                Console.WriteLine("\t\tElement CodeBase identyfikatora aplikacji: {0}",
                    info.ApplicationTrust.ApplicationIdentity.CodeBase);
                Console.WriteLine("\t\tPena nazwa identyfikatora aplikacji: {0}",
                    info.ApplicationTrust.ApplicationIdentity.FullName);
                Console.WriteLine("\t\tDefaultGrantSet: {0}",
                    info.ApplicationTrust.DefaultGrantSet.ToXml());
                Console.WriteLine("\t\tCzy aplikacja jest zaufana: {0}",
                    info.ApplicationTrust.IsApplicationTrustedToRun);
            }

            Console.WriteLine("\tcieka do kopii shadow aplikacji: {0}",
                info.CachePath);
            Console.WriteLine("\tPlik konfiguracyjny: {0}",
                info.ConfigurationFile);
            Console.WriteLine("\tPlik licencji tej domeny aplikacji mona znale tu: {0}",
                info.LicenseFile);

            Console.WriteLine("\tParametry adowania kompilata:");
            Console.WriteLine("\t\tCzy jest dozwolone testowanie cieek appbase oraz prywatnej cieki bin?: {0}",
                !info.DisallowApplicationBaseProbing);
            Console.WriteLine("\t\tPrywatna cieka do binariw: {0}",
                info.PrivateBinPath);
            if (info.PrivateBinPathProbe != null)
            {
                if (info.PrivateBinPathProbe.Length > 0)
                    Console.WriteLine("\t\tWyczenie katalogu AppBase ze cieki wyszukiwania");
                else
                    Console.WriteLine("\t\tWczenie katalogu AppBase do cieki wyszukiwania");
            }

            Console.WriteLine("\t\tCzy jest dozwolone przekierowywanie powiza?: {0}",
                !info.DisallowBindingRedirects);
            Console.WriteLine("\t\tCzy jest dozwolone pobieranie kodu: {0}",
                !info.DisallowCodeDownload);
            Console.WriteLine("\t\tCzy wykorzystuje si strategi wydawcy? : {0}",
                !info.DisallowPublisherPolicy);
            Console.WriteLine("\t\tPliki wygenerowane dynamicznie s zapisane w: {0}",
                info.DynamicBase);

            Console.WriteLine("\t\tOptymalizacja programu adujcego: {0}",
                info.LoaderOptimization.ToString());
            Console.WriteLine("\t\tKatalogi kopii shadow: {0}",
                info.ShadowCopyDirectories);
            Console.WriteLine("\t\tKatalogi kopii shadow: {0}",
                info.ShadowCopyFiles);
        }
		#endregion

        #region "8.17 Programowe podwyszanie priorytetu procesu"
        public static void TestBoostPriority()
        {
            //  Uruchomienie rekurencyjnej operacji dir na dysku c:
            ProcessStartInfo psi = new ProcessStartInfo("cmd.exe","/C\"dir c:\\ /S\"");
            // Nie wywietlamy okna.
            psi.CreateNoWindow = true;
            // Uruchomienie  procesu.
            Process p = Process.Start(psi);
            // Podniesienie priorytetu procesu do poziomu High.
            p.PriorityClass = ProcessPriorityClass.High;
        }

		#endregion

        #region "8.18 Analiza rodowiska wykonawczego w celu diagnozowania problemw"
        public static void TestEnvironment()
        {
            Console.WriteLine("Biece ustawienia rodowiska:");
            foreach (DictionaryEntry de in Environment.GetEnvironmentVariables())
            {
                Console.WriteLine("\t\tZmienna rodowiskowa {0}: Warto {1}", de.Key, de.Value);
            }
            Console.WriteLine("\tUstawienie zmiennej rodowiskowej MYENVVAR na warto {0}", "FOO");
            Environment.SetEnvironmentVariable("MYENVVAR", "FOO");
            Console.WriteLine("\tPobranie informacji o zmiennej rodowiskowej MYENVVAR: {0}", Environment.GetEnvironmentVariable("MYENVVAR"));

            Console.WriteLine("\tWiersz polecenia: {0}", Environment.CommandLine);
            Console.WriteLine("\tBiecy katalog: {0}", Environment.CurrentDirectory);

            Console.WriteLine("\tBiecy kod wyjcia: {0}",Environment.ExitCode);
            Console.WriteLine("\tNazwa NetBIOS komputera: {0}", Environment.MachineName);
            Console.WriteLine("\tInformacje o wersji systemu operacyjnego: {0}", Environment.OSVersion.VersionString);
            Console.WriteLine("\tLiczba procesorw: {0}", Environment.ProcessorCount);
            Console.WriteLine("\tBieca zawarto stosu wywoa: {0}", Environment.StackTrace);
            Console.WriteLine("\tKatalog systemowy: {0}",Environment.SystemDirectory);
            Console.WriteLine("\tNazwa domeny biecego uytkownika: {0}", Environment.UserDomainName);
            Console.WriteLine("\tCzy to jest interaktywny proces uytkownika?: {0}", Environment.UserInteractive);
            Thread.CurrentThread.Name = "Gwny wtek";
            Console.WriteLine("\tUytkownik, ktry rozpocz wtek ({0}): {1}", Thread.CurrentThread.Name,
                Environment.UserName);
            RunImpersonatedThread(); // Przykad Environment.UserName
            Console.WriteLine("\tWersja rodowiska CLR, w ktrym dziaa kod: {0}", Environment.Version);
            Console.WriteLine("\tIlo fizycznej pamici wykorzystywana przez ten obszar roboczy: {0}", Environment.WorkingSet);
        }

        [DllImport("advapi32.dll", SetLastError = true)]
        public static extern bool LogonUser(String lpszUsername, String lpszDomain, String lpszPassword,
            int dwLogonType, int dwLogonProvider, ref IntPtr phToken);

        [DllImport("kernel32.dll", SetLastError = true)]
        static extern bool CloseHandle(IntPtr hObject);

        const int LOGON32_LOGON_INTERACTIVE = 2;
        const int LOGON32_PROVIDER_DEFAULT = 0;

        public static void RunImpersonatedThread()
        {
            Console.WriteLine("RunImpersonatedThread executed with identity {0}", WindowsIdentity.GetCurrent().Name);
            IntPtr userToken = IntPtr.Zero;
            if (LogonUser("TestUser", "SKYWALKER", "password", LOGON32_LOGON_INTERACTIVE,
                    LOGON32_PROVIDER_DEFAULT, ref userToken))
            {
                WindowsImpersonationContext w =
                    WindowsIdentity.Impersonate(userToken);

                Thread t = new Thread(new ThreadStart(Impersonate));
                t.Name = "Spersonifikowany wtek";
                Console.WriteLine("Utworzono wtek na podstawie identyfikatora {0}", WindowsIdentity.GetCurrent().Name);
                t.Start();

                w.Undo();

                Console.WriteLine("Powrcono do identyfikatora {0}", WindowsIdentity.GetCurrent().Name);
            }
            else
            {
                Console.WriteLine("Proces LogonUser nie powid si - kod bdu: {0}",Marshal.GetLastWin32Error());
            }
        }

        public static void Impersonate()
        {
            Console.WriteLine("\tUytkownik, ktry rozpocz wtek ({0}): {1}", Thread.CurrentThread.Name,
                Environment.UserName);
        }
		#endregion
	}
}
